/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.soap.hosts;

import java.io.IOException;
import android.text.TextUtils;
import de.avm.android.tr064.exceptions.BaseException;
import de.avm.android.tr064.exceptions.DataMisformatException;
import de.avm.android.tr064.model.HostNameEntry;
import de.avm.android.tr064.model.UnsignedNumber;
import de.avm.android.tr064.soap.ISoapCredentials;

public class GetHostEntry extends HostsSoapHelper<HostNameEntry>
{
    private int mIndex;
	private String mMac = null;
	
	/**
	 * Gets host entry by index
	 * 
	 * @param index
	 * 		Index of entry
     * @throws IllegalArgumentException
     * 		illegal value of index
	 */
	public GetHostEntry(ISoapCredentials soapCredentials, int index)
	{
		super(soapCredentials);
		UnsignedNumber.validateUi2(index, "index");
		mIndex = index;
	}
	
	/**
	 * Gets host entry by MAC address
	 * 
	 * @param mac
	 * 		MAC of device to get info from
	 */
	public GetHostEntry(ISoapCredentials soapCredentials, String mac)
	{
		super(soapCredentials);
		mMac = mac;
	}

    public int getIndex()
    {
        return mIndex;
    }

    /**
     * @param index
     * 		Index of entry
     * @throws IllegalArgumentException
     * 		illegal value of index
     */
    public GetHostEntry setIndex(int index)
    {
        UnsignedNumber.validateUi2(index, "index");
        mIndex = index;
        mMac = null;
        return this;
    }

    public String getMac()
    {
        return mMac;
    }

    public GetHostEntry setMac(String mac)
    {
        mMac = mac;
        return this;
    }

    @Override
	public String getSoapMethodParameter()
	{
		return (TextUtils.isEmpty(mMac)) ?
				"<NewIndex>" + Integer.toString(mIndex) + "</NewIndex>" :
					"<NewMACAddress>" + mMac + "</NewMACAddress>";
	}

	@Override
	public HostNameEntry getQualifiedResult()
			throws DataMisformatException, BaseException, IOException
	{
		String input = getSoapBody();
		
		HostNameEntry result = new HostNameEntry((TextUtils.isEmpty(mMac)) ?
				getValueByName("NewMACAddress", input) : mMac);
		result.setIPAddress(getValueByName("NewIPAddress", input));
		result.setAddressSource(getValueByName("NewAddressSource", input));
		result.setLeaseTimeRemaining(getLongByName("NewLeaseTimeRemaining", input));
		result.setInterfaceType(getValueByName("NewInterfaceType", input));
		result.setActive(getBooleanByName("NewActive", input));
		result.setHostName(getValueByName("NewHostName", input));
		
		return result;
	}

	@Override
	public String getSoapMethod()
	{
		return (TextUtils.isEmpty(mMac)) ?
				"GetGenericHostEntry" : "GetSpecificHostEntry";
	}
}
